
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Thu Dec 02 15:35:52 2004
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief A brief description of this module
   *
   *  \version CRYS_HMAC.c#1:csrc:4
   *  \author adams
   *  \remarks Copyright (C) 2004 by Discretix Technologies Ltd.
   *           All Rights reserved
   */



/************* Include Files ****************/

#include "DX_VOS_Mem.h"
#include "CRYS.h"
#include "CRYS_HMAC_error.h"
#include "dx_hw_defs.h"
#include "crys_host_op_code.h"

 
/************************ Defines ***************************************/

/* canceling the PC-lint warning:
   Use of goto is deprecated */
/*lint --e{801} */

/************************ macros ****************************************/

/************************ Typedefs **************************************/



/************************ Global Data ***********************************/

/************* Private function prototype *******************************/


/************************ Public Functions ******************************/

/**
 * This function initializes the HMAC machine on the CRYS level.
 *
 * This function allocates and initializes the HMAC Context .
 * The function receives as input a pointer to store the context handle to HMAC Context. 
 *
 * If the key is larger then 64 bytes it performs on it a HASH operation.
 * then the function executed a HASH_init session and processes a HASH update 
 * on the Key XOR ipad and stores it on the context.
 *
 * the context. 
 * 
 * @param[in] ContextID_ptr - a pointer to the HMAC context buffer allocated by the user that
 *                       is used for the HMAC machine operation.
 *
 * @param[in] OperationMode - The operation mode : MD5 or SHA1.
 *
 * @param[in] key_ptr - The pointer to the users key buffer.
 *
 * @oaram[in] keySize - The size of the received key.
 *
 * @return CRYSError_t on success the function returns CRYS_OK else non ZERO error.
 *      
 */
CEXPORT_C CRYSError_t CRYS_HMAC_Init(CRYS_HMACUserContext_t *ContextID_ptr,
                           CRYS_HASH_OperationMode_t         OperationMode,
                           DxUint8_t                        *key_ptr,
                           DxUint16_t                        keySize )
{
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_HMAC_INIT_OP_CODE, 4,
	   							 /* PARAMETERS: */
								 ContextID_ptr,
								 OperationMode,
								 key_ptr, keySize );
  
}/* END OF CRYS_HMAC_Init */

/**
 * This function process a HMAC block of data via the HASH Hardware.
 * The function receives as input an handle to the  HMAC Context , and operated the HASH update on the data
 * below.
 * 
 * @param[in] ContextID_ptr - a pointer to the HMAC context buffer allocated by the user that
 *                       is used for the HMAC machine operation.
 *
 * @param[in] DataIn_ptr - The pointer to the buffer of the input data to the HMAC. The pointer does 
 *                         not need to be aligned. On CSI input mode the pointer must be equal to
 *                         value (0xFFFFFFFC | DataInAlignment). 
 * 
 * @param[in] DataInSize - The size of the data to be hashed in bytes. On CSI data transfer mode the size must  
 *                         multiple of HASH_BLOCK_SIZE for used HASH mode.
 *
 * @return CRYSError_t - On success CRYS_OK is returned, on failure a
 *                        value MODULE_* CRYS_HMAC_error.h
 *
 */

CEXPORT_C CRYSError_t CRYS_HMAC_Update(CRYS_HMACUserContext_t  *ContextID_ptr,
                             DxUint8_t						   *DataIn_ptr,
                             DxUint32_t                        DataInSize )
{
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_HMAC_UPDATE_OP_CODE, 3,
	   							 /* PARAMETERS: */
								 ContextID_ptr,
								 DataIn_ptr, DataInSize );

}/* END OF CRYS_HMAC_Update */ 

/**
 * This function finalize the HMAC process of data block.
 * The function receives as input an handle to the HMAC Context , that was initialized before
 * by an CRYS_HMAC_Init function or by CRYS_HMAC_Update function.
 * The function finishes the HASH operation on the ipad and text then 
 * executes a new hash operation with the key XOR opad and the previous HASH operation result.  
 *
 *  @param[in] ContextID_ptr - a pointer to the HMAC context buffer allocated by the user that
 *                       is used for the HMAC machine operation.
 *
 *  @retval HmacResultBuff a pointer to the target buffer where the 
 *                       HMAC result stored in the context is loaded to.
 *
 *  @return CRYSError_t on success the function returns CRYS_OK else non ZERO error.
 */

CEXPORT_C CRYSError_t CRYS_HMAC_Finish( CRYS_HMACUserContext_t  *ContextID_ptr ,
                              CRYS_HASH_Result_t       HmacResultBuff )
{
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_HMAC_FINISH_OP_CODE, 3,
	   							 /* PARAMETERS: */
								 ContextID_ptr,
								 HmacResultBuff );
  
}/* END OF CRYS_HMAC_Finish */

/**
 * @brief This function is a service function that frees the context if the operation has
 *        failed. 
 *
 *        The function executes the following major steps:
 *
 *        1. Checks the validity of all of the inputs of the function. 
 *           If one of the received parameters is not valid it shall return an error.
 *
 *           The major checkers that are run over the received parameters:
 *           - verifying that the context pointer is not DX_NULL (*ContextID_ptr). 
 *        2. Clearing the users context.
 *        3. Exit the handler with the OK code.
 *      
 *
 * @param[in] ContextID_ptr - a pointer to the HMAC context buffer allocated by the user that
 *                       is used for the HMAC machine operation. this should be the same context that was
 *                       used on the previous call of this session.
 *
 *
 * @return CRYSError_t - On success CRYS_OK is returned, on failure a
 *                        value MODULE_* CRYS_HMAC_error.h
 */

CEXPORT_C CRYSError_t  CRYS_HMAC_Free(CRYS_HMACUserContext_t  *ContextID_ptr )
{
  /* The return error identifier */
  CRYSError_t Error;

  /*--------------------
      CODE
  -----------------------*/
  Error = CRYS_OK;
  
  /* ............... checking the parameters validity ................... */
   
  /* if the users context ID pointer is DX_NULL return an error */
  if( ContextID_ptr == DX_NULL )
  {
    Error = CRYS_HMAC_INVALID_USER_CONTEXT_POINTER_ERROR;
    goto end_function;
  }
   
  /* .............. clearing the users context .......................... */
  /* -------------------------------------------------------------------- */
   
  DX_VOS_MemSet( ContextID_ptr , 0 , sizeof(CRYS_HMACUserContext_t) );   

  /* ................. end of function ..................................... */
  /* ----------------------------------------------------------------------- */
 
end_function:
      
  return Error;   
 
}/* END OF CRYS_HMAC_Free */

/**
 * This function provide HASH function to process one buffer of data.
 * The function allocates an internal HASH Context , it initializes the 
 * HASH Context with the cryptographic attributes that are needed for 
 * the HASH block operation ( initialize H's value for the HASH algorithm ).
 * Then the function loads the Hardware with the initializing values and after 
 * that process the data block using the hardware to do hash .
 * At the end the function return the message digest of the data buffer .
 *
 *
 * @param[in] OperationMode - The operation mode : MD5 or SHA1.
 *
 * @param[in] key_ptr - The pointer to the users key buffer.
 *
 * @oaram[in] keySize - The size of the received key.
 * 
 * @param[in] ContextID_ptr - a pointer to the HMAC context buffer allocated by the user that
 *                       is used for the HMAC machine operation.
 *
 * @param[in] DataIn_ptr - The pointer to the buffer of the input data to the HMAC. The pointer does 
 *                         not need to be aligned. On CSI input mode the pointer must be equal to
 *                         value (0xFFFFFFFC | DataInAlignment). 
 * 
 * @param[in] DataInSize - The size of the data to be hashed in bytes. On CSI data transfer mode the size must  
 *                         multiple of HASH_BLOCK_SIZE for used HASH mode.
 *
 * param[out] HashResultBuff - a pointer to the target buffer where the 
 *                      HMAC result stored in the context is loaded to.
 *
 * @return CRYSError_t on success the function returns CRYS_OK else non ZERO error.
 *
 */
CEXPORT_C CRYSError_t CRYS_HMAC( 
                         CRYS_HASH_OperationMode_t     OperationMode,
                         DxUint8_t                    *key_ptr,
                         DxUint16_t                    keySize,
                         DxUint8_t                    *DataIn_ptr,
                         DxUint32_t                    DataSize,
                         CRYS_HASH_Result_t            HmacResultBuff )
{                         
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_HMAC_OP_CODE, 6,
	   							 /* PARAMETERS: */
								 OperationMode,
								 key_ptr, keySize,
								 DataIn_ptr, DataSize,
								 HmacResultBuff );
}/* END OF CRYS_HMAC */   

